/*
 * Scrollbar.java Version 0.11 99/11/07 
 *
 * History: 
 *  1999-10-22 Version number 0.1 released
 *  1999-11-07 Version number 0.11 released
 *	       - problem in paint fixed
 *		 (sometimes resizing causes an 
 *		  ArtithmeticException)
 *
 *  1999-12-14 SH removed double paint problem for List
 *
 * Copyright 1999 by Stefan Haustein & Michael Kroll
 * 
 * stefan.haustein@trantor.de
 * michael.kroll@trantor.de
 *
 * All rights reserved.
 *
 * NO WARRANTY
 *
 *  1. BECAUSE THE PROGRAM IS LICENSED FREE OF CHARGE, THERE IS NO
 *  WARRANTY FOR THE PROGRAM, TO THE EXTENT PERMITTED BY APPLICABLE
 *  LAW.  EXCEPT WHEN OTHERWISE STATED IN WRITING THE COPYRIGHT
 *  HOLDERS AND/OR OTHER PARTIES PROVIDE THE PROGRAM "AS IS" WITHOUT
 *  WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING, BUT
 *  NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 *  FITNESS FOR A PARTICULAR PURPOSE.  THE ENTIRE RISK AS TO THE
 *  QUALITY AND PERFORMANCE OF THE PROGRAM IS WITH YOU.  SHOULD THE
 *  PROGRAM PROVE DEFECTIVE, YOU ASSUME THE COST OF ALL NECESSARY
 *  SERVICING, REPAIR OR CORRECTION.
 * 
 *  2. IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW OR AGREED TO IN
 *  WRITING WILL ANY COPYRIGHT HOLDER, OR ANY OTHER PARTY WHO MAY
 *  MODIFY AND/OR REDISTRIBUTE THE PROGRAM AS PERMITTED ABOVE, BE
 *  LIABLE TO YOU FOR DAMAGES, INCLUDING ANY GENERAL, SPECIAL,
 *  INCIDENTAL OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OR
 *  INABILITY TO USE THE PROGRAM (INCLUDING BUT NOT LIMITED TO LOSS OF
 *  DATA OR DATA BEING RENDERED INACCURATE OR LOSSES SUSTAINED BY YOU
 *  OR THIRD PARTIES OR A FAILURE OF THE PROGRAM TO OPERATE WITH ANY
 *  OTHER PROGRAMS), EVEN IF SUCH HOLDER OR OTHER PARTY HAS BEEN
 *  ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 * 
 *  END OF TERMS AND CONDITIONS */

package java.awt;

import java.awt.event.*;

/** <ul>
 *    <li>Only one Listener for each event Type is allowed
 *    <li>AdjustmentEvents are delivered directly to the Listener
 *       without going through ProcessAdjustmentEvent!
 *   </ul>
 */


public class Scrollbar extends Component {  
	
    public static final int HORIZONTAL = 0;
    public static final int VERTICAL   = 1;

    static Image imgUp = new Image (7, 6, 1,   "0010387cfe00");
    static Image imgDown = new Image (7, 6, 1, "00fe7c381000");
    static Image imgUpI = new Image (7, 6, 1,  "feeec68200fe");
    static Image imgDownI = new Image (7, 6, 1,"fe0082c6eefe");
    
    static Image imgLeft = new Image (6, 7, 1,   "08183878381808");
    static Image imgRight = new Image (6, 7, 1,  "40607078706040");
    static Image imgLeftI = new Image (6, 7, 1,  "f4e4c484c4e4f4");
    static Image imgRightI = new Image (6, 7, 1, "bc9c8c848c9cbc");
    
    int minW = 7;
    int minH = 12;
    
    int minValue;
    int maxValue;
    int currValue;
    int orientation;
    int currVisible;	
    
    int scrPos;
    int scrBarSize;
    int scrSpace;
    
    int unitIncrement = 1;
    int blockIncrement;

    int scrollType = -1;

    /*
      boolean pressingInc = false;
      boolean pressingDec = false;
    */

    boolean dragging = false;
    
    AdjustmentListener adjustmentListener;
    Component internalOwner;
    
    public Scrollbar () {
	this (VERTICAL);
    }

    public Scrollbar (int orientation) {
	this (orientation, 0, 10, 0, 100);
    } 

    public Scrollbar (int orientation, int value, int visible, 
		      int minimum, int maximum) {
 		
	this.orientation = orientation;
	this.currValue = value;
	this.currVisible = visible;
	this.minValue = minimum;
	this.maxValue = maximum;
	
	blockIncrement = currVisible;
    }
    
    
    public void addAdjustmentListener (AdjustmentListener al) {
	if (adjustmentListener != null) 
	    throw new TooManyListenersException ();
	
	adjustmentListener = al;
    }
    
    
    public int getMaximum() {
   	return maxValue;
    }
    
    
    public int getMinimum() {
  	return minValue;
    }                   
    
    
    public Dimension getMinimumSize () {
	if (orientation == VERTICAL)
	    return new Dimension (minW, minH);
	else 
	    return new Dimension (minH, minW);
    }
    

    public int getBlockIncrement () {
	return blockIncrement;
    }
    

    public int getOrientation() {
      return orientation;
    }

    
    public int getUnitIncrement() {
	return unitIncrement;
    }
                   
    
    public int getValue() {
   	return currValue;
    }

    
    public void paint(Graphics g) {
	
	scrSpace = ((orientation == Scrollbar.HORIZONTAL) 
		    ? w : h) - 12;
	
	scrBarSize = (scrSpace * currVisible) 
	    / (maxValue - minValue + currVisible);

	if (scrBarSize < 4) scrBarSize = 4;
	
	scrSpace -= scrBarSize;

	scrPos = 6 + (scrSpace * (currValue - minValue)) 
	    / (maxValue - minValue);

    	if (orientation != Scrollbar.HORIZONTAL) { // VERT
	    
	    int middle = w / 2;
	    
	    g.drawImage ((scrollType == AdjustmentEvent.UNIT_DECREMENT) 
			 ? imgUpI : imgUp, middle -3, 0, null);
	    
	    g.drawImage ((scrollType == AdjustmentEvent.UNIT_INCREMENT) 
			 ? imgDownI : imgDown, middle - 3, h - 6, null);

	    g.setColor(Color.gray);

	    if (Graphics.monochrome) {
		g.drawLine(middle-1, 6, middle-1, h-7);
		g.drawLine(middle, 7, middle, h-8);
		g.drawLine(middle+1, 6, middle+1, h-7);
	    }
	    else g.fillRect (middle-1, 6, 3, h-12);
		
	    g.setColor (Color.black);
	    g.fillRect (middle-1, scrPos, 3, scrBarSize);

	    g.setColor (Color.white);

	    g.drawLine (middle-1, scrPos-1, middle+1, scrPos-1); 

	    g.drawLine (middle-1, scrPos + scrBarSize, 
			middle+1, scrPos + scrBarSize); 
    	} 
	else {
	    int middle = h / 2;
          
	    g.drawImage ((scrollType == AdjustmentEvent.UNIT_DECREMENT) 
			 ? imgLeftI : imgLeft, 0, middle-3, null);
	    
	    g.drawImage ((scrollType == AdjustmentEvent.UNIT_INCREMENT) 
			 ? imgRightI : imgRight, w - 6, middle - 3, null);

	    g.setColor(Color.gray);
          	
	    if (Graphics.monochrome) {
		g.drawLine(6, middle-1, w-7, middle-1);
		g.drawLine(6, middle, w-7, middle);
		g.drawLine(6, middle+1, w-7, middle+1);
		g.setXORMode(Color.black);          	
		g.drawLine(6, middle, w-7, middle);
		g.setPaintMode();          	
	    }
	    else g.fillRect (6, middle-1, w-12, 3);
	    
	    g.setColor(Color.black);
	    g.fillRect (scrPos, middle-1, scrBarSize, 3);
	    
	    g.setColor (Color.white);
	    g.drawLine (scrPos, middle-1, scrPos, middle+1);
	    g.drawLine (scrPos+scrBarSize, middle -1, 
			scrPos+scrBarSize, middle+1);
	}
    }
    
    /*
      public void processAdjustmentEvent (AdjustmentEvent e) {
      if (adjustmentListener != null)
      adjustmentListener.adjustmentValueChanged (e);
      }
    */
    
    public void processMouseEvent (MouseEvent e) {

	int id = e.getID ();
	int mpos = (orientation != HORIZONTAL) ? e.getY () : e.getX ();
	boolean befBar = mpos <= scrPos + scrBarSize;
	boolean behBar = mpos >= scrPos;
	/*
	pressingDec = false;
	pressingInc = false;
	*/
	int newVal = currValue;
	scrollType = -1;

	if (id == e.MOUSE_PRESSED && befBar && behBar)
	    dragging = true;
	else if (id == e.MOUSE_DRAGGED && dragging) {
	    
	    newVal = minValue 
		+ ((mpos - 7 - scrBarSize / 2) 
		   * (maxValue-minValue)) / scrSpace;

	    scrollType = AdjustmentEvent.TRACK;
	}
	else if (id == e.MOUSE_DRAGGED || id == e.MOUSE_PRESSED) {
	    dragging = false;

	    if (befBar) {

		if (mpos <= 4) {
		    newVal = currValue - unitIncrement;
		    scrollType = AdjustmentEvent.UNIT_DECREMENT;
		}
		else {
		    newVal = currValue - blockIncrement; 
		    scrollType = AdjustmentEvent.BLOCK_DECREMENT;
		}
	    }
	    else {
		if ((mpos >= (h - 4) 
		     && orientation != HORIZONTAL)
		    || (mpos >= (w - 4) && orientation == HORIZONTAL)) {

		    newVal = currValue + unitIncrement; 
		    scrollType = AdjustmentEvent.UNIT_INCREMENT;
		}
		else {
		    newVal = currValue + blockIncrement;
		    scrollType = AdjustmentEvent.BLOCK_INCREMENT;
		}
	    }
	}
	else if (id == e.MOUSE_RELEASED) {
	    if (internalOwner == null) 
		paint (getGraphics ());
	    else internalOwner.repaint (x, y, w, h);
	}

	if (newVal > maxValue) newVal = maxValue;
	if (newVal < minValue) newVal = minValue;

	if (currValue != newVal) {
	    currValue = newVal;

	    if (internalOwner == null) paint (getGraphics ());
	    else internalOwner.repaint ();
	    
	    if (adjustmentListener != null)
		adjustmentListener.adjustmentValueChanged 
		    (new AdjustmentEvent 
			(this, AdjustmentEvent.ADJUSTMENT_VALUE_CHANGED,
			 scrollType, currValue));
	}
    }


    public void setBlockIncrement (int inc) {
	blockIncrement = inc;
    }
    
    
    public void setMaximum (int max) {
	maxValue = max;
	repaint ();
    }
    
    public void setMinimum (int min) {
	minValue = min;
	repaint ();
    }
    
    public void setOrientation (int orientation) {
	this.orientation = orientation;
	valid = false;
    }
    
    
    public void setUnitIncrement (int inc) {
	unitIncrement = inc;
    }
    
    
    public void setValue (int v) {
	currValue = v;
	repaint ();
    }
    

    public void setValues (int val, int visible, int min, int max) {
	maxValue = max;
	minValue = min;
	currVisible = visible;
	currValue = val;
	repaint ();
    }
}
